"""
Docker Hub OAuth2 backend, docs at:
    https://python-social-auth.readthedocs.io/en/latest/backends/docker.html
"""

from typing import Any

from .oauth import BaseOAuth2


class DockerOAuth2(BaseOAuth2):
    name = "docker"
    ID_KEY = "user_id"
    AUTHORIZATION_URL = "https://hub.docker.com/api/v1.1/o/authorize/"
    ACCESS_TOKEN_URL = "https://hub.docker.com/api/v1.1/o/token/"
    REFRESH_TOKEN_URL = "https://hub.docker.com/api/v1.1/o/token/"
    REDIRECT_STATE = False
    EXTRA_DATA = [
        ("refresh_token", "refresh_token", True),
        ("user_id", "user_id"),
        ("email", "email"),
        ("full_name", "fullname"),
        ("location", "location"),
        ("url", "url"),
        ("company", "company"),
        ("gravatar_email", "gravatar_email"),
    ]

    def get_user_details(self, response):
        """Return user details from Docker Hub account"""
        fullname, first_name, last_name = self.get_user_names(
            response.get("full_name") or response.get("username") or ""
        )
        return {
            "username": response.get("username"),
            "fullname": fullname,
            "first_name": first_name,
            "last_name": last_name,
            "email": response.get("email", ""),
        }

    def user_data(self, access_token: str, *args, **kwargs) -> dict[str, Any] | None:
        """Grab user profile information from Docker Hub."""
        username = kwargs["response"]["username"]
        return self.get_json(
            f"https://hub.docker.com/api/v1.1/users/{username}/",
            headers={"Authorization": f"Bearer {access_token}"},
        )
